#include "stdafx.h"
#include "utilities.h"
#include "ReMapper.h"
#include "float.h"

static float vertWeldThreshold = 0.0f;
static float uvWeldThreshold = 0.0f;
static BOOL duplicateBackfaceVerts = FALSE;
static BOOL weldVerticesOnPosition = TRUE;

ReMapper::ReMapper(void)
{
    m_numVertices = 0;
    m_numFaces = 0;
    m_vertices = NULL;
    m_faces = NULL;    
}

ReMapper::~ReMapper(void)
{
    m_numVertices = 0;

    if (m_vertices)
    {
        RwFree(m_vertices);
        m_vertices = NULL;
    }

    if (m_faces)
    {
        RwFree(m_faces);
        m_faces = NULL;
    }
}

static int
#ifdef _WINDOWS
__cdecl
#endif /* _WINDOWS */
ReMapperInfo_Compare(const void *pA,const void *pB)
{
    ReMapperInfo *mapA = (ReMapperInfo *)pA;
    ReMapperInfo *mapB = (ReMapperInfo *)pB;

    assert(mapA);
    assert(mapB);

    /* can we sort on vertex positions? */
    if (mapA->vertexIndex != mapB->vertexIndex)
    {
        if (!weldVerticesOnPosition)
        {
            return (mapA->vertexIndex - mapB->vertexIndex);
        }
        if (mapA->x - mapB->x < -vertWeldThreshold)
        {
            return (-1);
        }
        else if (mapA->x - mapB->x > vertWeldThreshold)
        {
            return (1);
        }
        if (mapA->y - mapB->y < -vertWeldThreshold)
        {
            return (-1);
        }
        else if (mapA->y - mapB->y > vertWeldThreshold)
        {
            return (1);
        }
        if (mapA->z - mapB->z < -vertWeldThreshold)
        {
            return (-1);
        }
        else if (mapA->z - mapB->z > vertWeldThreshold)
        {
            return (1);
        }
    }

    /* can we sort on texture UV's? */    
    if (mapA->u - mapB->u < -uvWeldThreshold)
    {
        return (-1);
    }
    else if (mapA->u - mapB->u > uvWeldThreshold)
    {
        return (1);
    }

    if (mapA->v - mapB->v < -uvWeldThreshold)
    {
        return (-1);
    }
    else if (mapA->v - mapB->v > uvWeldThreshold)
    {
        return (1);
    }

    /* can we sort on smoothing groups */    
    if (mapA->usedSMGroup == 0 ||
        mapB->usedSMGroup == 0)
    {
        /*  if there aren't any smoothing groups we duplicate
            to get same results as max */
        return (1);
    }
        
    if (mapA->usedSMGroup > mapB->usedSMGroup)
    {
        return (1);
    }
    else if (mapA->usedSMGroup < mapB->usedSMGroup)
    {
        return (-1);
    }    
    
    /* unshare verts across material boundaries.
       Needed for Pipeline II but shouldn't hurt PI much */
    
    if (mapA->matID > mapB->matID)
    {
        return (1);
    }
    else if (mapA->matID < mapB->matID)
    {
        return (-1);
    }    

    if (duplicateBackfaceVerts)
    {
        if (mapA->backface && !mapB->backface)
        {
            return (1);
        }
        else if (mapB->backface && !mapA->backface)
        {
            return (-1);
        }
    }

       /* can we sort on colors? */
    if (mapA->colorIndex != mapB->colorIndex)
    {
        if (mapA->r > mapB->r)
        {
            return (1);
        }
        else if (mapA->r < mapB->r)
        {
            return (-1);
        }
        if (mapA->g > mapB->g)
        {
            return (1);
        }
        else if (mapA->g < mapB->g)
        {
            return (-1);
        }
        if (mapA->b > mapB->b)
        {
            return (1);
        }
        else if (mapA->b < mapB->b)
        {
            return (-1);
        }
    }

        
    /* must be equal then */
    return (0);
}

void
ReMapper::InitReMapperInfos(ReMapperInfo *infos, int numInfos)
{
    for (int i=0; i<numInfos; i++)
    {
        infos[i].backface = FALSE;
        infos[i].faceIndex = -1;
        infos[i].origFace = -1;
        infos[i].smGroup = 0;
        infos[i].usedSMGroup = 0;
        infos[i].textureIndex = -1;
        infos[i].triIndex = -1;
        infos[i].u = 0.0f;
        infos[i].v = 0.0f;
        infos[i].vertexIndex = -1;
        infos[i].x = 0.0f;
        infos[i].y = 0.0f;
        infos[i].z = 0.0f;
        infos[i].matID = -1;
        infos[i].r = 0;
        infos[i].g = 0;
        infos[i].b = 0;
        infos[i].colorIndex = -1;
    }
}

BOOL 
ReMapper::DoRemap(Mesh *mesh, Mtl *mat, BOOL limitUVs, int uvMax, float weldThreshold,
                  float weldThresholdUV, BOOL export2SidedMaterials,
                  BOOL maintainBackfaceNormals, BOOL weldVertices,
                  WarningList *warnings)
{
    ReMapperInfo *remapper;
    int faceNum, faceIndex;
    int numNonDegenFaces, backfaceindex = 1;
    int numFaces = mesh->getNumFaces();
    int numNonBackfacedFaces = 0;
    float uOffset = 0.0f, vOffset = 0.0f;
    int i;
    bool GeneratedTilingError = false;
    
    /* Setup user weld thresholds */
    vertWeldThreshold = weldThreshold;
    uvWeldThreshold = weldThresholdUV;
    /* setup whether to duplicate verts for backfaces */
    duplicateBackfaceVerts = maintainBackfaceNormals;
    weldVerticesOnPosition = weldVertices;

    if (mesh->getNumTVerts() > 0 && mesh->tvFace)
    {
        /* Calculate per mesh uv offsets to get into a positive right
           way up mapping */
        CalcTextureOffsets(mesh, &uOffset, &vOffset, mat);
    }
    
    numNonDegenFaces = NumNonDegenerateFaces(mesh, mat, weldThreshold, export2SidedMaterials, weldVertices);
    m_numFaces = numNonDegenFaces;
    remapper = (ReMapperInfo *)RwMalloc(sizeof(ReMapperInfo) * numNonDegenFaces * 3);    
    if (!remapper)
    {
        return (FALSE);
    }
    InitReMapperInfos(remapper, numNonDegenFaces * 3);

    /* Start calculations for smoothing groups, we need vertices to know which
       smoothing groups they belong to and not just faces */
    m_vertSMGroups = (unsigned int *)RwMalloc(sizeof(unsigned int) * mesh->numVerts);
    m_vertSMGroupsTemp = (unsigned int *)RwMalloc(sizeof(unsigned int) * mesh->numVerts);
    for (i = 0; i<mesh->numVerts; i++)
    {
        m_vertSMGroupsTemp[i] = 0;
        m_vertSMGroups[i] = 0;
    }
    for (faceIndex = 0; faceIndex < numFaces; faceIndex++)
    {
        int j;
        for (j = 0; j < 3; j++)
        {
            m_vertSMGroups[mesh->faces[faceIndex].v[j]] |= m_vertSMGroupsTemp[mesh->faces[faceIndex].v[j]] & mesh->faces[faceIndex].smGroup;
            m_vertSMGroupsTemp[mesh->faces[faceIndex].v[j]] |= mesh->faces[faceIndex].smGroup;            
        }
    }
    RwFree(m_vertSMGroupsTemp);

    faceIndex = 0;
    m_currentMap = remapper;
    for (faceNum = 0; faceNum < numFaces; faceNum++)
    {
        int j;        
        ReMapperFaceLoop *triFace = NULL;
        ReMapperFaceLoop *texFace = NULL;
        
        //determine whether the texture on this face is tiled or not
        //generate an error if u tiling xor v tiling (unsupported).
        bool UTiling, VTiling, UMirror, VMirror;
        getMtlTiling( mat, mesh->getFaceMtlIndex(faceNum),
                    &UTiling, &VTiling, &UMirror, &VMirror );
        if ((UTiling^VTiling)&!GeneratedTilingError) {
            warnings->add(CWarning(wtError,"","Illegal tiling detected in material (u and v must have the same options)"));
            GeneratedTilingError=true;
        }

        if (!DegenerateFace(mesh, faceNum, weldThreshold, weldVertices))
        {
            float minu = FLT_MAX, minv = FLT_MAX;
            Matrix3 UVTransform;
            UVVert transformedUV[3];

            m_limituvoffsetu=0.0f;
            m_limituvoffsetv=0.0f;
            if (mesh->getNumTVerts() > 0 && mesh->tvFace)
            {
                int MatID = mesh->getFaceMtlIndex(faceNum);
                GetMaxUVTransform(mat, MatID, &UVTransform);
                            
                //transform the uv coordinates
                for (j = 0; j < 3; j++)
                {
                    UVVert* texVert = mesh->getTVertPtr(mesh->tvFace[faceNum].t[j]);
                    transformedUV[j] = *texVert * UVTransform;
                }

                if (isMtlCropped(mat, MatID))
                {
                    for (j = 0; j < 3; j++)
                    {                        
                        /* find min uv on this face */                        
                        if (transformedUV[j].x < minu)
                        {
                            minu = transformedUV[j].x;
                        }
                        if (transformedUV[j].y < minv)
                        {
                            minv = transformedUV[j].y;
                        }                        
                    }
                     
                    /* really just want to subtract floor(minu) from the u's etc
                       but have to guard against floor of a 1.0 float being 0.0
                       since it's actually 0.99999995 */
                    if (minu - (float)floor(minu) > 0.99999f)
                    {
                        minu = (float)floor(minu) + 1.0f;
                    }
                    else
                    {
                        minu = (float)floor(minu);
                    }
                    if (minv - (float)floor(minv) > 0.99999f)
                    {
                        minv = (float)floor(minv) + 1.0f;
                    }
                    else
                    {
                        minv = (float)floor(minv);
                    }
                    for (j = 0; j < 3; j++)
                    {
                        transformedUV[j].x -= minu;
                        if (transformedUV[j].x < 0.0f) transformedUV[j].x = 0.0f;
                        transformedUV[j].y -= minv;
                        if (transformedUV[j].y < 0.0f) transformedUV[j].y = 0.0f;
                    }

                    for (j = 0; j < 3; j++)
                    {
                        ApplyCroppingToUVs(mat, mesh->getFaceMtlIndex(faceNum), &transformedUV[j]);                  
                    }
                }
                
                /* apply the per mesh offset and find minimum uv on this face */
                minu = FLT_MAX; minv = FLT_MAX;
                for (j=0; j<3; j++)
                {
                    /* apply the per mesh offset, depending on whether tiling is enabled */
                    if (UTiling) {
                        transformedUV[j].x = uOffset + transformedUV[j].x;
                    }
                    if (VTiling) {
                        transformedUV[j].y = vOffset - transformedUV[j].y;
                    }
                    else {
                        transformedUV[j].y = 1 - transformedUV[j].y;
                    }
                    
                    /* find minimum uv */
                    if (transformedUV[j].x < minu)
                    {
                        minu = transformedUV[j].x;
                    }
                    if (transformedUV[j].y < minv)
                    {
                        minv = transformedUV[j].y;
                    }
                }
                
                /* keep uv's within a specified 0->x limit if required */
                if (limitUVs)
                {                    
                    if (minu > uvMax)
                    {
                        m_limituvoffsetu = (float)fmod(minu, uvMax) - (float)minu;
                    }
                    if (minv > uvMax)
                    {
                        m_limituvoffsetv = (float)fmod(minv, uvMax) - (float)minv;
                    }
                }
            }

            WriteCurrentFace( false,faceIndex,backfaceindex,faceNum,
                                mesh,transformedUV );
            numNonBackfacedFaces++;

            if (export2SidedMaterials &&
                DoubleSidedMaterial(mat, mesh->faces[faceNum].getMatID()))
            {
                /* map in the backface */
                WriteCurrentFace( true,faceIndex,backfaceindex,faceNum,
                                    mesh,transformedUV );
                backfaceindex++;
            }
            faceIndex++;
        }
    }

    /* sort on vertexIndex->colorIndex->normalIndex */
    qsort(remapper, numNonDegenFaces * 3, sizeof(ReMapperInfo), ReMapperInfo_Compare);

    /* now remap the vertex face triangles */
    {
        int mapNum;
        int currentVertexIndex = -1;
    
        m_faces = (ReMapperFaceLoop *)RwMalloc(sizeof(ReMapperFaceLoop) * numNonDegenFaces);
        if (!m_faces)
        {
            RwFree(remapper);
            return (false);
        }

        for (mapNum = 0; mapNum < numNonDegenFaces * 3; mapNum++)
        {
            int faceIndex, triIndex, origFace;

            /* do we increment the vertex index? */
            if ((mapNum == 0) || 
                (ReMapperInfo_Compare(&remapper[mapNum-1], &remapper[mapNum]) != 0))
            {
                currentVertexIndex++;
            }

            /* remap its associated face */            
            faceIndex = remapper[mapNum].faceIndex;
            origFace = remapper[mapNum].origFace;
            if (remapper[mapNum].backface)
            {
                faceIndex = numNonBackfacedFaces + remapper[mapNum].backface - 1;
            }
            triIndex = remapper[mapNum].triIndex;
            m_faces[faceIndex].index[triIndex] = currentVertexIndex;
            m_faces[faceIndex].faceNum = origFace;
        }

        m_numVertices=currentVertexIndex+1;
        m_vertices = (ReMapperVertex *)RwMalloc(sizeof(ReMapperVertex) * m_numVertices);
        if (!m_vertices)
        {
            RwFree(m_faces);
            RwFree(remapper);
            return (FALSE);
        }

        currentVertexIndex = -1;
        
        for (mapNum = 0; mapNum < numNonDegenFaces * 3; mapNum++)
        {
            if ((mapNum == 0) || 
                (ReMapperInfo_Compare(&remapper[mapNum-1], &remapper[mapNum]) != 0))
            {
                int vertexIndex, textureIndex, faceIndex;

                currentVertexIndex++;

                /* get vertexIndex */
                vertexIndex = remapper[mapNum].vertexIndex;
                m_vertices[currentVertexIndex].vertexIndex = vertexIndex;
                
                /* get textureIndex */
                textureIndex = remapper[mapNum].textureIndex;
                m_vertices[currentVertexIndex].textureIndex = textureIndex;
                m_vertices[currentVertexIndex].u = remapper[mapNum].u;
                m_vertices[currentVertexIndex].v = remapper[mapNum].v;

                /* get smGroup */
                m_vertices[currentVertexIndex].smGroup = remapper[mapNum].smGroup;

                /* get vertex color index */
                m_vertices[currentVertexIndex].colorIndex = remapper[mapNum].colorIndex;

                /* get faceIndex */
                faceIndex = remapper[mapNum].origFace;                
                m_vertices[currentVertexIndex].faceIndex = faceIndex;
                m_vertices[currentVertexIndex].backface = remapper[mapNum].backface;
                
            }
        }
    }

    /* done its job */
    RwFree(remapper);
    RwFree(m_vertSMGroups);

    /* all done beautifully */
    return (TRUE);
}

void ReMapper::WriteCurrentFace( bool isBackFace,
                                 int faceIndex,
                                 int backfaceindex,
                                 int faceNum,
                                 Mesh *mesh,
                                 UVVert transformedUV[3] )
{
    int j, newindex;

    for (j = 0; j < 3; j++)
    {
        newindex=j;
        //Swap vertices around if this is a back face
        if (isBackFace) {
            if (j==0) newindex = 1;
            else if (j==1) newindex = 0;
        }

        m_currentMap->faceIndex = faceIndex;
        m_currentMap->matID = mesh->faces[faceNum].getMatID();
        m_currentMap->origFace = faceNum;
        m_currentMap->triIndex = j;                
        m_currentMap->smGroup = mesh->faces[faceNum].smGroup;       
        m_currentMap->usedSMGroup = mesh->faces[faceNum].smGroup &
                                m_vertSMGroups[mesh->faces[faceNum].v[newindex]];

        if (mesh->faces)
        {
            m_currentMap->vertexIndex = mesh->faces[faceNum].v[newindex];
            m_currentMap->x = mesh->verts[m_currentMap->vertexIndex].x;
            m_currentMap->y = mesh->verts[m_currentMap->vertexIndex].y;
            m_currentMap->z = mesh->verts[m_currentMap->vertexIndex].z;
        }

        if (mesh->tvFace)
        {
            m_currentMap->textureIndex = mesh->tvFace[faceNum].t[newindex];
            m_currentMap->u = transformedUV[newindex].x + m_limituvoffsetu;
            m_currentMap->v = transformedUV[newindex].y + m_limituvoffsetv;
        }
                
        if (mesh->vcFace)
        {
            m_currentMap->colorIndex = mesh->vcFace[faceNum].t[newindex];
            m_currentMap->r = (RwUInt8)(mesh->vertCol[m_currentMap->colorIndex].x * 255.0f);
            m_currentMap->g = (RwUInt8)(mesh->vertCol[m_currentMap->colorIndex].y * 255.0f);
            m_currentMap->b = (RwUInt8)(mesh->vertCol[m_currentMap->colorIndex].z * 255.0f);
        }

        if (isBackFace)
            m_currentMap->backface = backfaceindex;

        m_currentMap++;
    }
}

void
ReMapper::ResetReMap(void)
{
    m_numVertices = 0;
    m_numFaces = 0;

    if (m_vertices)
    {
        RwFree(m_vertices);
        m_vertices = NULL;
    }

    if (m_faces)
    {
        RwFree(m_faces);
        m_faces = NULL;
    }
}

ReMapperFaceLoop *
ReMapper::GetFace(int faceNum)
{
    if (m_faces)
    {
        return (&m_faces[faceNum]);
    }

    return (NULL);
}

ReMapperVertex *
ReMapper::GetVertex(int vertexNum)
{
    if (m_vertices)
    {
        return (&m_vertices[vertexNum]);
    }

    return (NULL);
}

int
ReMapper::GetNumVertices(void)
{
    return (m_numVertices);
}

int
ReMapper::GetNumFaces(void)
{
    return (m_numFaces);
}
